#!/usr/bin/perl
use strict;
#
# Script to create a bootable sys$rom.bin image for Psion 5mx Pro
# Copyright 2002 Tony Lindgren <tony@atomide.com>
#

# Boot loader code start offset
my $BOOTSTART = 0xf3c;

my $IMAGE = "sys\$rom.bin";

my $BOOTBIN = "proboot.bin";
my $LINUX_TEXTADDR = 0xc0048000;
my $LINUXSTART = (($LINUX_TEXTADDR << 2) >> 2);
my $MEMBASE = (($LINUX_TEXTADDR >> 28) << 28);
my $PARAMSTART = $LINUXSTART - (128 * 1024);

# We place the initrd at 2 megs from the start.
my $INITRD_ADDR = 0xc0200000;
my $INITRDSTART = (($INITRD_ADDR << 2) >> 2);
my $INITRDSIZE = 0;

# These must match setup.h
my $ATAG_CORE= 0x54410001;
my $ATAG_INITRD = 0x54410005;
my $ATAG_CMDLINE = 0x54410009;
my $ATAG_NONE = 0x00000000;

#
# Main program
#
my ($LINUXIMAGE, $LINUXINITRD, $LINUXPARAMS) = @ARGV;

if ($LINUXIMAGE eq "") {
	print_usage();
}

if ($LINUXINITRD eq "-" || $LINUXINITRD eq "none") {
	$LINUXINITRD = "";
}

print "Using kernel image: ".$LINUXIMAGE."\n";
print "Using kernel parameters: ".$LINUXPARAMS."\n";
print "Using initrd: ".$LINUXINITRD."\n";

$INITRDSIZE = -s $LINUXINITRD;
print "Initrd size: ".$INITRDSIZE."\n";

#
# First we need to create a 256 byte header for the
# image that works with the Psion boot loader. We
# only need one ARM instruction there to tell where
# to jump to start the proboot execution.
#
create_header($IMAGE);
my $imagesize = -s $IMAGE;
pad_file($IMAGE, ($BOOTSTART - $imagesize), 0);

#
# Append the proboot binary to the image
#
concat_files($IMAGE, $BOOTBIN);

#
# Pad until we are at the Linux parameter address
# then write the parameters
#
$imagesize = -s $IMAGE;
pad_file($IMAGE, ($PARAMSTART - $imagesize));
write_param_tags($IMAGE, $LINUXPARAMS);

#
# Pad until we are at the Linux TEXTADDR, and then append
# the Linux kernel image
#
$imagesize = -s $IMAGE;
pad_file($IMAGE, ($LINUXSTART - $imagesize));
concat_files($IMAGE, $LINUXIMAGE, 0);

#
# Finally append the initrd
#
if ($LINUXINITRD ne "") {
	$imagesize = -s $IMAGE;
	if ($INITRDSTART < $imagesize) {
	  die "ERROR: Initrd overlaps with the kernel image\n";
	}
	pad_file($IMAGE, ($INITRDSTART - $imagesize));
	concat_files($IMAGE, $LINUXINITRD);	
} else {
	print "No initrd added to the boot image\n";
}

print "\nBoot image creation done. The new boot image is ".$IMAGE."\n\n";

exit;

#
# Subroutines
#
sub create_header() {
	my($file) = @_;
	open OUT, ">$file"
		or die "ERROR: Cannot open file ".$file;
	print "Writing boot image header\n";
	print OUT "\0\0\0\0";
	# Write out some ARM assembly: cc 03 00 ea
	print OUT "\xcc\x03\x00\xea";
	close (OUT);
}

sub concat_files() {
	my($file1, $file2) = @_;
	open OUT, ">>$file1"
		or die "ERROR: Cannot open file ".$file1;
	open IN, "<$file2"
		or die "ERROR: Cannot open file ".$file2;
	print "Appending ".$file2." to ".$file1."\n";
	while (<IN>) {
		print OUT;
	}
	close (IN);
	close (OUT);
	my $concat_size = -s $file1;
	print "The new size is: ".$concat_size."\n";
}

sub pad_file() {
	my($file, $padsize, $overwrite) = @_;
	if ($overwrite gt 0) {
		print"Overwriting ".$file."\n";
		open OUT, ">$file";
	} else {
		open OUT, ">>$file";
	}

	print "Padding ".$file." with ".$padsize." bytes\n";
	for (my $i=0; $i < $padsize; $i++) {
		print OUT "\0";
	}
	close (OUT);
	my $padded_size = -s $file;
	print "The new size is: ".$padded_size."\n";
}

sub write_param_tags() {
	my($file, $commandline) = @_;
	write_atag_core($file);
	write_atag_cmdline($file, $commandline);
	if ($LINUXINITRD ne "") {
		write_atag_initrd($file, ($INITRDSTART | $MEMBASE), $INITRDSIZE);
	}
	write_atag_endtag($file);
}

sub write_atag_core() {
	my($file) = @_;
	my $atag_core_size = (4 + 4 + 4);
	write_atag_header($file, $atag_core_size, $ATAG_CORE);
	open OUT, ">>$file"
		or die "Cannot open file ".$file;
	print OUT pack('L', 0);
	print OUT pack('L', 0);
	print OUT pack('L', 0);
	close OUT;
}

sub write_atag_endtag() {
	my ($file) = @_;
	open OUT, ">>$file"
		or die "Cannot open file ".$file;
	print OUT pack('L', 0);
	print OUT pack('L', $ATAG_NONE);
	close OUT;
}

sub write_atag_cmdline() {
	my($file, $commandline) = @_;
	my $cmd_length = (length($commandline));
	write_atag_header($file, $cmd_length, $ATAG_CMDLINE);
	open OUT, ">>$file"
		or die "Cannot open file ".$file;
	print OUT $commandline;
	close OUT;	
}

sub write_atag_initrd() {
	my($file, $start, $size) = @_;
	my $initrd_tag_size = (4 + 4);
	write_atag_header($file, $initrd_tag_size, $ATAG_INITRD);
	write_atag_mem($file, $start, $size);
}

sub write_atag_header() {
	my($file, $size, $tag) = @_;
	my $header_size = 4 + 4;
	open OUT, ">>$file"
		or die "Cannot open file ".$file;
	print OUT pack('L', (($size + $header_size) >> 2));
	print OUT pack('L', $tag);
	close OUT;
}

sub write_atag_mem() {
	my($file, $start, $size) = @_;
	open OUT, ">>$file"
		or die "Cannot open file ".$file;	
	print OUT pack('L', $start);
	print OUT pack('L', $size);
	close OUT;
}

sub print_usage() {
  print "\n";
  print "Usage:  ".$0." linuximage initrd params\n\n";

  print "  Example 1: Root on CF card, console on LCD\n";
  print "  ".$0." Image - \"root=/dev/hda2 ro console=tty0\"\n\n";

  print "  Example 2: Root on CF card, console on LCD and serial port\n";
  print "  ".$0." Image - \"root=/dev/hda2 ro console=tty0 console=ttyAM1,115200\"\n\n";

  print "  Example 3: Root on initrd, console on LCD and serial port\n";
  print "  ".$0." Image myinitrd.gz \"root=/dev/ram0 ro console=tty0 console=ttyAM1,115200\"\n";
  print "\n";
  exit;
}
